import { ApiProperty, ApiPropertyOptional } from "@nestjs/swagger";
import { IsNumber, IsObject, IsOptional, IsString } from "class-validator";
import { Column, CreateDateColumn, Entity, JoinColumn, ManyToOne, OneToMany, OneToOne, PrimaryGeneratedColumn, Unique, UpdateDateColumn } from "typeorm";
import { Product } from "../../products/entities/product.entity";
import { FileEntity } from "../../files/entities/file.entity";
import { Optional } from "@nestjs/common";

@Entity("categories")
@Unique(["name"])
export class Category {
    @PrimaryGeneratedColumn()
    id?: number;

    @Column()
    @ApiProperty({ example: 'category name', description: 'The name of the category' })
    @IsOptional()
    @IsString()
    name: string;

    @Column({ nullable: true })
    @ApiPropertyOptional({ example: 'category description', description: 'The description of the category' })
    @IsOptional()
    @IsString()
    description?: string;

    @Column({ name: "icon_id", nullable: true })
    @ApiPropertyOptional({ description: "file id to upload" })
    @IsOptional()
    @IsString()
    iconId?: string;
    @OneToOne(() => FileEntity, (file) => file.category, { cascade: ['insert', 'update'] })
    @JoinColumn({ name: "icon_id" })
    icon?: Partial<FileEntity>;

    @OneToMany(() => Category, (category) => category.parentCategory, { cascade: ['insert', 'update'] })
    @JoinColumn({ name: 'sub_category_id' })
    @ApiPropertyOptional({ type: [Category], description: 'The sub categories of the category' })
    subCategories?: Partial<Category>[];

    @ManyToOne(() => Category, (category) => category.subCategories,
        { cascade: true, onDelete: 'CASCADE', nullable: true })
    @JoinColumn({ name: "parent_category_id" })
    @ApiPropertyOptional({ type: Category, description: 'The parent category of the category' })
    @IsOptional()
    @IsObject()
    parentCategory?: Partial<Category>;

    @OneToMany(() => Product, (product) => product.category)
    products?: Product[];


    @CreateDateColumn()
    createdAt?: Date;

    @UpdateDateColumn()
    updated_at?: Date;
}
