import { ApiProperty, ApiPropertyOptional } from "@nestjs/swagger";
import { IsEnum, IsNotEmpty, IsNumber, IsOptional, IsString } from "class-validator";
import { Column, CreateDateColumn, Entity, JoinColumn, ManyToOne, OneToMany, PrimaryGeneratedColumn, UpdateDateColumn } from "typeorm";
import { ProductStatus } from "../enums";
import { Category } from "../../category/entities/category.entity";
import { User } from "../../users/entities/user.entity";
import { FileEntity } from "../../files/entities/file.entity";
import { Brand } from "../../brands/entities/brand.entity";

@Entity("product")
export class Product {
    @PrimaryGeneratedColumn()
    id?: number;

    @Column()
    @ApiProperty({ example: 'product name', description: 'The name of the product' })
    @IsString()
    name: string;

    @Column({ nullable: true })
    @ApiPropertyOptional({ example: 'product description', description: 'The description of the product' })
    @IsOptional()
    @IsString()
    description: string;

    @Column()
    @ApiPropertyOptional({ example: 'product price', description: 'The price of the product' })
    @IsNumber()
    price: number;

    @Column()
    @ApiPropertyOptional({ example: 'product quantity', description: 'The quantity of the product' })
    @IsNumber()
    quantity: number;

    @Column({ default: 0 })
    @ApiPropertyOptional({ example: 'product discount', description: 'The discount of the product' })
    @IsOptional()
    @IsNumber()
    discount?: number;

    @Column({ type: "enum", enum: ProductStatus, default: ProductStatus.InStock })
    @ApiPropertyOptional({ example: 'product status', description: 'The status of the product' })
    @IsOptional()
    @IsEnum(ProductStatus)
    status: ProductStatus;


    @Column({ name: "category_id" })
    @ApiProperty({ example: 1, description: 'The category of the product' })
    @IsNumber()
    categoryId?: number;
    @ManyToOne(() => Category, (category) => category.products)
    @JoinColumn({ name: "category_id" })
    category: Partial<Category> | number;

    @ManyToOne(() => Brand, (brand) => brand.products)
    @JoinColumn({ name: "brand_id" })
    @ApiProperty({ example: 1, description: 'The brand of the product' })
    @IsNumber()
    brand: Partial<Brand> | number;

    //images (file entity)
    @OneToMany(() => FileEntity, (file) => file.product, { cascade: ['insert', 'update'] })
    @ApiPropertyOptional({ example: [], description: "file to upload" })
    @IsOptional()
    images: Partial<FileEntity>[] | string[];

    @ManyToOne(() => User, (user) => user.productsCreated)
    @JoinColumn({ name: "created_by_id" })
    createdBy: Partial<User> | number;


    @CreateDateColumn()
    createdAt?: Date;

    @UpdateDateColumn()
    updated_at?: Date;
}
